#include "globals.h"
#include "ProcessorSpecific.h"
//#include "ESPfile.h"

/******************************************************************************
 * MACROS
*******************************************************************************/
#define PtoA(page) 		(FAT_ADDRESS+PAGE_SIZE*(page))								// Page_to_Address
#define PtoS(page)		(FAT_SECTOR+(u16)(PAGE_SIZE*(page)/BLOCK_SIZE))				// Page_to_Sector
#define StoA(sector)	((sector)*BLOCK_SIZE+me_correct)							// Sector_to_Address
#define S1P(sector)		((u16) ((sector)-FAT_SECTOR)*BLOCK_SIZE/PAGE_SIZE)			// Sector_1st_Page
/*===========================================================================*/


/*********************************************************************************/
enum p_mode{
	CLOSED,
	READ,
	WRITE,
	READ_WRITE,
	APPEND
};

struct {
	s8 pipe_id;														// to which pipeN: attached
	u8 intf_num;													// interface number pipe attached to
	u16 file_index;													// FAT file index
    enum p_mode mode;												// see above
}LOCAL pipesAttachedFile[PIPE_COUNT]={0};							/// Every TCP pipe goes here

/*======================	HELP FUNCTIONS	============================================*/

/******************************************************************************
* FunctionName : checkSum(intf_number, params)
* Description  : calculates check sum of file
* Parameters   : u8 *
*				  May be done using CRC later!
* Returns      : u16 check sum
*******************************************************************************/
u16 checkSum(u8 intf_num){
	u16 mysum = 0, len = FATtable[myInterfaces[intf_num].id].length;
	u8 *p = (fileMountAddress[intf_num].ping>0) ? fileMountAddress[intf_num].ping : fileMountAddress[intf_num].pong;
	while (len){
		mysum += *p++;
		len--;
	}
	return mysum;
}
/*======================	PIPE FUNCTIONS	============================================*/

/******************************************************************************
 * FunctionName : fileAddPipe(u8 pipe, u8 intf_num, u8 *params)
 * Description  : attach a pipe to a open file
 * Parameters   :
 *
 * Returns      :
*******************************************************************************/
void ICACHE_FLASH_ATTR fileAddPipe(u8 pipe, u8 intf_num, u8 *params){
	u8 i, new_pipe = 0xff, same_intf = 0xff;
	const char *HELP="{interfaceN}";

	if(find_char(params, '?')) {toP0((u8*) HELP,0,0); return;}

	if (myInterfaces[intf_num].state == 0) return;									// file not open
	for (i = 0; i<PIPE_COUNT; i++){
		if (pipesAttachedFile[i].pipe_id<0){
			new_pipe = i;
			break;
		}
	}
	if (new_pipe==0xff) {toP0((u8*) "No free pipe", 0, 0); return;}								// no free pipe
	pipesAttachedFile[new_pipe].file_index = myInterfaces[intf_num].id;
	pipesAttachedFile[new_pipe].intf_num = intf_num;
	myInterfaces[intf_num].pipes_attached++;
	pipesAttachedFile[new_pipe].mode = READ_WRITE;
	pipesAttachedFile[new_pipe].pipe_id = pipe;
	myStreams[pipe].pipe_count++;
	myStreams[pipe].rxPipe.val = (fileMountAddress[intf_num].ping>0) ? fileMountAddress[intf_num].ping : fileMountAddress[intf_num].pong;
	myStreams[pipe].txPipe.val = myStreams[pipe].rxPipe.val;
	if (myInterfaces[intf_num].subtype == 1){
		myStreams[pipe].txPipe.len = &FATtable[myInterfaces[intf_num].id].length;
		myStreams[pipe].rxPipe.len = &FATtable[myInterfaces[intf_num].id].length;
	}
	else {
		myStreams[pipe].txPipe.len = &fileSize[intf_num];
		myStreams[pipe].rxPipe.len = &fileSize[intf_num];
	}
    bitset(&myPipes,pipe);														// pipe OK
	myInterfaces[intf_num].state = INTF_CONNECT;
}

/******************************************************************************
 * FunctionName : fileRemovePipe(u8 pipe, u8 intf_num)
 * Description  : unattach a pipe from file
 * Parameters   : pipeN:, intf_num
 *
 * Returns      :
*******************************************************************************/
void ICACHE_FLASH_ATTR fileRemovePipe(u8 pipe, u8 intf_num){
	u8 i;
	for (i = 0; i<PIPE_COUNT; i++){
		if (pipesAttachedFile[i].pipe_id==pipe && pipesAttachedFile[i].intf_num==intf_num) break;
	}
	if (i != PIPE_COUNT){
		pipesAttachedFile[i].pipe_id = -1;
		pipesAttachedFile[i].mode = CLOSED;
		freePipeCheck(pipe);
		if(--(myInterfaces[intf_num].pipes_attached)<=0){
			myInterfaces[intf_num].pipes_attached=0;
			myInterfaces[intf_num].state=INTF_OPEN;
		}
	}
}

/*======================	INTERFACE FUNCTIONS	============================================*/

/******************************************************************************
 * FunctionName : initFS()
 * Description  : call once at beginning to fill FAT
 * Parameters   :
 *
 * Returns      :
*******************************************************************************/
void ICACHE_FLASH_ATTR initFS(){
	u16 i;
	t_file *fat = FATtable, *f;
	u8 *map = FATmap, *m;
	u8 *FAT = (u8*) os_malloc(FAT_PAGES_USED * PAGE_SIZE);

	for (i = 0; i<PIPE_COUNT; i++){
		pipesAttachedFile[i].pipe_id = -1;
		pipesAttachedFile[i].mode = CLOSED;
	}

	for (i = 0; i<INTERFACES_NUM; i++){
		fileMountAddress[i].ping=0;
		fileMountAddress[i].pong=0;
	}

	if (spi_flash_read(FAT_ADDRESS, (u32*)FAT, FAT_PAGES_USED * PAGE_SIZE) != SPI_FLASH_RESULT_OK) { toP0((u8*) "File system not mount", 0, 0); return; }
	f = (t_file*) FAT;
	m = FAT + FAT_TABLE_LEN;
	for (i = 0; i < FILE_NUM; i++){									// FAT table fill in
		*fat++ = *f++;
	}
	for (i = 0; i < FAT_MAP_LEN; i++){								// FAT map fill in
		*map++ = *m++;
	}

	for(i=0; i<FAT_PAGES_USED; i++){
			FATmap[i]=roEMPTY-2;									// mark FAT pages as File(0xef or 0xfd). Easy to format FS first time
	}
	my_free(FAT);
}

/******************************************************************************
* FunctionName : listDir()
* Description  : print a file list
* Parameters   :
*
* Returns      :
*******************************************************************************/
u16  ICACHE_FLASH_ATTR listDir(){
	u8 i, list[NAME_LEN+10];
	u8* lp = list;
	u32 sum_len = 0;
	for (i = 0; i < FILE_NUM; i++){
		if (FATtable[i].name[0] != roEMPTY){
			my_memset(lp, ' ', NAME_LEN+1);
			lp[NAME_LEN + 1] = 0;
			fl_memcpy(lp, FATtable[i].name, my_strlen(FATtable[i].name));
			add_char(list, '%');
			add_char(list, 'j');
			toP0(list, 0, FATtable[i].length);
			sum_len += FATtable[i].length;
		}
	}
	return toP0((u8*) "total: %ik from %jk", sum_len >> 10, FS_SIZE >> 10);
}

/******************************************************************************
 * FunctionName : closeFile(u8 intf_num)		ex: U1{0}
 * Description  : save file to FLASH. Clean or add pages used by file. If file empty - delete it.
 * Parameters   : intf_numer	number of U
 *
 * Returns      :
*******************************************************************************/
void  ICACHE_FLASH_ATTR fileClose(u8 intf_num){
	u16 j, sector, curPage, curSector;
	s32 writelen=FATtable[myInterfaces[intf_num].id].length;
	u16 seek=0;
	u8 *blockP, *source;
	u16 gap = 0, source_gap = 0;

	if (myInterfaces[intf_num].state == INTF_CLOSE) return;
	for (j=0;j<PIPE_COUNT;j++){
		if(pipesAttachedFile[j].pipe_id>=0 && pipesAttachedFile[j].intf_num==intf_num)					// check if pipe belongs to interface
		{
			fileRemovePipe(pipesAttachedFile[j].pipe_id, intf_num);										// and close all them

		}
	}
	if (myInterfaces[intf_num].subtype == 2){
		if (fileMountAddress[intf_num].ping){
			my_free(fileMountAddress[intf_num].ping);
			fileMountAddress[intf_num].ping = 0;
		}
		if (fileMountAddress[intf_num].pong){
			my_free(fileMountAddress[intf_num].pong);
			fileMountAddress[intf_num].pong = 0;
		}
		myInterfaces[intf_num].state = INTF_CLOSE;
		return;
	}
	ETS_FRC1_INTR_DISABLE();
	ETS_UART_INTR_DISABLE();
	if (writelen == 0){
		my_memset(FATtable[myInterfaces[intf_num].id].name, roEMPTY, NAME_LEN);							// Delelte file
		FATtable[myInterfaces[intf_num].id].attr.state = 0;
		FATtable[myInterfaces[intf_num].id].checksum = 0;
	}
	else FATtable[myInterfaces[intf_num].id].checksum = checkSum(intf_num);
	curPage=0;
	source = (fileMountAddress[intf_num].ping>0) ? fileMountAddress[intf_num].ping : fileMountAddress[intf_num].pong;
	for(sector=FAT_SECTOR; sector<SECTOR_LIMIT; sector++){												// run check across all sectors
		gap=0;
		for(j=0; j<BLOCK_SIZE/PAGE_SIZE; j++){															// BLOCK_SIZE/PAGE_SIZE gives pages per block
			curPage=S1P(sector)+j;
			if (writelen > 0){
				if (FATmap[curPage] == roEMPTY || FATmap[curPage] == myInterfaces[intf_num].id){				// if found empty page or page already belongs to this file
					blockP = (u8*) my_malloc(BLOCK_SIZE);
					if (blockP == NULL) { toP0((u8*)"Out of RAM", 0, 0); return; }
					spi_flash_read(StoA(sector), (u32*)blockP, BLOCK_SIZE);							// read whole block of 4k
					spi_flash_erase_sector(sector);
					curPage = S1P(sector);																// rewind to 1-st page
					while (curPage < S1P(sector + 1)){														// run across the block pages
						if (writelen > 0){
							if (FATmap[curPage] == roEMPTY || FATmap[curPage] == myInterfaces[intf_num].id){
								//u16 mylen = (writelen < PAGE_SIZE) ? writelen : PAGE_SIZE;
								FATmap[curPage] = myInterfaces[intf_num].id;
								fl_memcpy(blockP + gap, source + source_gap, PAGE_SIZE);
								source_gap += PAGE_SIZE;
								writelen -= PAGE_SIZE;
							}
						}
						else{
							if (FATmap[curPage] == myInterfaces[intf_num].id){							// clear all the file's pages in this sector
								FATmap[curPage] = roEMPTY;
								my_memset(blockP + gap, roEMPTY, PAGE_SIZE);
							}
						}
						curPage++;
						gap += PAGE_SIZE;
					}
					spi_flash_write(StoA(sector), (u32*)blockP, BLOCK_SIZE);							// restore cleaned block
					my_free((char*)blockP);
					break;
				}
			}
			else{
				if(FATmap[curPage]==myInterfaces[intf_num].id){											// chunk of file found in next sectors
					blockP = (u8*) my_malloc(BLOCK_SIZE);
					if(blockP==NULL) {toP0((u8*)"Out of RAM",0,0);return;}
					spi_flash_read(StoA(sector), (u32*) blockP, BLOCK_SIZE);							// read whole block of 4k
					spi_flash_erase_sector(sector);
					curPage=S1P(sector);

					while(curPage < S1P(sector+1)){
						if(FATmap[curPage]==myInterfaces[intf_num].id){
							FATmap[curPage]=roEMPTY;
							my_memset(blockP+gap, roEMPTY, PAGE_SIZE);
						}
						curPage++;
						gap+=PAGE_SIZE;
					}
					spi_flash_write(StoA(sector), (u32*) blockP, BLOCK_SIZE);							// restore cleaned block
					my_free((char*) blockP);
					break;																				// force next sector
				}
			}
		}
	}
	if (fileMountAddress[intf_num].ping){
		my_free(fileMountAddress[intf_num].ping);
		fileMountAddress[intf_num].ping = 0;
	}
	if (fileMountAddress[intf_num].pong){
		my_free(fileMountAddress[intf_num].pong);
		fileMountAddress[intf_num].pong = 0;
	}														//

	// Last we need to save FAT
	blockP = (u8*) my_malloc(BLOCK_SIZE);
	if(blockP==NULL) {toP0((u8*)"Out of RAM",0,0);return;}
	my_memset(blockP, roEMPTY, FAT_PAGES_USED*PAGE_SIZE);													// set FAT to zero
	spi_flash_read(StoA(FAT_SECTOR), (u32*) blockP, BLOCK_SIZE);									// read whole block of 4k
	spi_flash_erase_sector(FAT_SECTOR);
	fl_memcpy(blockP,(u8*) FATtable, sizeof(FATtable));
	fl_memcpy(blockP+sizeof(FATtable), FATmap, sizeof(FATmap));
	spi_flash_write(StoA(FAT_SECTOR), (u32*) blockP, BLOCK_SIZE);							// restore cleaned block
	my_free((char*) blockP);
	myInterfaces[intf_num].state=INTF_CLOSE;
	ETS_FRC1_INTR_ENABLE();
	ETS_UART_INTR_ENABLE();
}

/******************************************************************************
 * FunctionName : fileResize(u8 intf_num)		ex: U1{0}
 * Description  : called instead of sendInterface in user task (user_main) do resize RAM used by files
 * Parameters   : intf_numer	number of U
 *
 * Returns      :
*******************************************************************************/
void  ICACHE_FLASH_ATTR fileResize(u8 intf_num){
	u8 i,j;
	u16 new_size, f_len;
	if (intf_num == INTERFACES_NUM - 2) return;															// if mule.ml no need to be resized
	// DISABLE ALL INTERRUPTS!
	ETS_FRC1_INTR_DISABLE();
	ETS_UART_INTR_DISABLE();
	for (i = 0; i<PIPE_COUNT; i++){
		if (pipesAttachedFile[i].pipe_id >= 0 && pipesAttachedFile[i].intf_num == intf_num){
			if (myInterfaces[intf_num].subtype == 1){
				f_len = FATtable[myInterfaces[intf_num].id].length;
				if ((fileSize[intf_num] - f_len < PAGE_SIZE * 2) || (fileSize[intf_num] - f_len > PAGE_SIZE * 6)){				// if just 2 Pages left or more than 6 Pages empty
					new_size = f_len + 4 * PAGE_SIZE;														// set size + 4 pages
					if (fileMountAddress[intf_num].pong == 0){
						fileMountAddress[intf_num].pong = (u8*)my_malloc(new_size);
						if (fileMountAddress[intf_num].pong == 0) break;
						fl_memcpy(fileMountAddress[intf_num].pong, fileMountAddress[intf_num].ping, f_len);
						my_free(fileMountAddress[intf_num].ping);
						fileMountAddress[intf_num].ping = 0;
						fileSize[intf_num] = new_size;
					}
					else if (fileMountAddress[intf_num].ping == 0){
						fileMountAddress[intf_num].ping = (u8*)my_malloc(new_size);
						if (fileMountAddress[intf_num].ping == 0) break;
						fl_memcpy(fileMountAddress[intf_num].ping, fileMountAddress[intf_num].pong, f_len);
						my_free(fileMountAddress[intf_num].pong);
						fileMountAddress[intf_num].pong = 0;
						fileSize[intf_num] = new_size;
					}
				}
				if (fileMountAddress[intf_num].ping > 0){
					myStreams[pipesAttachedFile[i].pipe_id].rxPipe.val = fileMountAddress[intf_num].ping;			// if some file resized this will spread change to all the pipes attached to it
					myStreams[pipesAttachedFile[i].pipe_id].txPipe.val = fileMountAddress[intf_num].ping;			// if not will do nothing wrong
				}
				else{
					myStreams[pipesAttachedFile[i].pipe_id].rxPipe.val = fileMountAddress[intf_num].pong;			// if some file resized this will spread change to all the pipes attached to it
					myStreams[pipesAttachedFile[i].pipe_id].txPipe.val = fileMountAddress[intf_num].pong;			// if not will do nothing wrong
				}
			}
			//else{
			//	u16 txlen = *myStreams[pipesAttachedFile[i].pipe_id].txPipe.len;
			//	u16 rxlen = *myStreams[pipesAttachedFile[i].pipe_id].rxPipe.len;
			//	if (rxlen > 0 && txlen > rxlen){
			//		fl_memcpy(fileMountAddress[intf_num], fileMountAddress[intf_num] + rxlen, txlen - rxlen);
			//		*myStreams[pipesAttachedFile[i].pipe_id].txPipe.len -= (txlen - rxlen);
			//		*myStreams[pipesAttachedFile[i].pipe_id].rxPipe.len = 0;
			//	}
			//}
		}
	}
	// ENABLE ALL INTERRUPTS!
	ETS_FRC1_INTR_ENABLE();
	ETS_UART_INTR_ENABLE();
}

/******************************************************************************
 * FunctionName : openFile(u8 intf_num, u8 *params)
 * Description  : call when open U<intf_num>{type, sub-type, "name[NAME_LEN=12]"}	ex: U1{6,1,"my_file1"}
 * Parameters   : intf_numer	number of U
 *				 *params		type=6 /file/, sub-type=1/2 (ROM/RAM), "name"-string
 * Returns      :
*******************************************************************************/
void  ICACHE_FLASH_ATTR openFile(u8 intf_num, u8 *params){						// Ux{6,1,"name",(optional) size, comment_mode 0/1}
	u8* myp=params;
	u8 sub_type;																// 1-st get sub_type
	u8 index=0xff;																// file index in FAT
	u8 f_name[NAME_LEN];
	u16 f_size = PAGE_SIZE*4;													// if file size not present, create file as 4 pages long. Later it will be resizet as need
	u8 f_mode = 0;																// if file mode not set, open/create in 'include comments' mode i.e.'as is'
	const char *HELP="{6,1/2=File/RAMstream, if(File) \"name[%i]\" if(stream) size in bytes, if(File) (opt.)size, 0/1 skip_comments}";

	if(find_char(params, '?')) {toP0((u8*) HELP,NAME_LEN-1,0); return;}

	sub_type=indexE(&myp,10);													// 1-st get sub_type
	m_len = 0;
	mode.f.M = 0;
	myp++;
	if(sub_type==1){															// ROM file
		u8 i, f_name_len, empty_index=0xff;
		t_file *f= FATtable;
		//if(*myp=='?'){toP0((u8*)"Ux{6,1-ROM file,\"name[%i]\"(opt.), size, 0/1 skip_comments}",NAME_LEN,0); return;}
		Ex(&myp,0);																// get file name
		//if (my_strstr(out$.val, (u8*) ".ml") > 0) mode.f.M = 1;					// f_mode=0 - common file/ 1- .ml file need to cut comments, \r\n
		f_name_len=out$.len>=NAME_LEN ? NAME_LEN-1 : out$.len;					// cut to length 6
		fl_memcpy(f_name, out$.val, f_name_len);
		f_name[f_name_len] = 0;
		f_name_len++;															// to copy \0 too later
		if (*myp == ','){
			m_len = indexE(&myp, 10);											// u16 file size + 4 Pages
			f_size += m_len;
			if (*myp == ','){
				f_mode = indexE(&myp, 10);
				mode.f.M= (f_mode>0) ? 1 : 0;
			}
		}

		for(i=0; i<FILE_NUM; i++){
			if ((f + i)->name[0] == roEMPTY){
				if (empty_index == 0xff) empty_index = i;								// find first empty file descriptor
			}
			else if (my_strstr(f_name, (f + i)->name) > 0) {								// file found
				index = i; break;
			}
		}
		if(index==0xff){																	// must create file
			if(empty_index == 0xff)	return;												// no free file descriptor
			index=empty_index;
			fileMountAddress[intf_num].ping = (u8*) my_malloc(f_size);						// set pointer to a RAM area size of 2PAGE_SIZE
			if (fileMountAddress[intf_num].ping == NULL) { toP0((u8*)"Out of memory", 0, 0); return; }
			fileSize[intf_num] = f_size;
			my_memset(fileMountAddress[intf_num].ping, roEMPTY, f_size);					// clear in ROM sense
			fl_memcpy((f+index)->name, f_name, f_name_len);
			(f+index)->length=0;
			(f+index)->checksum=0;
			(f+index)->attr.state=1;														// file open, no pipes attached TODO
			myInterfaces[intf_num].state = INTF_CREATE;
		}
		else{
			u16 j;
			s32 readlen = FATtable[index].length;
			u8 *readP;
			if (intf_num == INTERFACES_NUM - 2) f_size = ((u16)((readlen + PAGE_SIZE - 1) / PAGE_SIZE))*PAGE_SIZE;	// if mule.ml open as is fit to PageSize
			else f_size += readlen;															// else open with own length + 4 pages just in case
			fileMountAddress[intf_num].ping=(u8*) my_malloc(f_size);						// set pointer to a RAM area size of file_length + 4 PAGE_SIZE
			if(fileMountAddress[intf_num].ping==NULL) {toP0((u8*)"Out of memory",0,0);return;}
			fileSize[intf_num] = f_size;
			my_memset(fileMountAddress[intf_num].ping, roEMPTY, f_size);					// clear in ROM sense
			// Read file to RAM
			readP=fileMountAddress[intf_num].ping;
			for(j=0; (j<FAT_MAP_LEN) && (readlen>0); j++){									// run across FATmap
				if(FATmap[j]==index){														// if page belongs to file, read in RAM
					spi_flash_read(PtoA(j), (u32*) readP, PAGE_SIZE);
					readlen-= PAGE_SIZE;
					readP += PAGE_SIZE;
				}
			}
			myInterfaces[intf_num].id=index;
			if (readlen > 0 || FATtable[index].checksum!=checkSum(intf_num)) {
				toP0((u8*)"Bad file", 0, 0);
				myInterfaces[intf_num].state = INTF_ERR;
				my_free(fileMountAddress[intf_num].ping);
				fileMountAddress[intf_num].ping = 0;
			}
			else myInterfaces[intf_num].state = INTF_OPEN;
		}
		myInterfaces[intf_num].id=index;										// id=index shoes sectors mark in sector map and also index in fileMountAddress[]. 0xff- means not used.
		myInterfaces[intf_num].closeInterface =fileClose;						// call ex: x=(*uart0_close)(0)
		myInterfaces[intf_num].sendInterface=fileResize;
		myInterfaces[intf_num].openPipe=fileAddPipe;
		myInterfaces[intf_num].closePipe=fileRemovePipe;
		myInterfaces[intf_num].type=6;
		myInterfaces[intf_num].subtype=1;
	}
	else if(sub_type==2){														// RAM Cyclic Stream Ux{6,2,size}
		u16 f_size;
		//if (*myp == '?') { toP0((u8*)"Ux{6,2-RAM buffer,size}", 0 , 0); return; }
		f_size= indexE(&myp, 10);
		fileMountAddress[intf_num].ping = (u8*)my_malloc(f_size);						// set pointer to a RAM area size of 2PAGE_SIZE
		if (fileMountAddress[intf_num].ping == NULL) { toP0((u8*)"Out of memory", 0, 0); return; }
		fileSize[intf_num] = 0;
		myInterfaces[intf_num].id = 0xff;										// id=index shoes sectors mark in sector map and also index in FATtable. 0xff- means not FAT alocated.
		myInterfaces[intf_num].closeInterface = fileClose;						// call ex: x=(*uart0_close)(0)
		myInterfaces[intf_num].sendInterface = fileResize;
		myInterfaces[intf_num].openPipe = fileAddPipe;
		myInterfaces[intf_num].closePipe = fileRemovePipe;
		myInterfaces[intf_num].type = 6;
		myInterfaces[intf_num].subtype = 2;
		myInterfaces[intf_num].state = INTF_CREATE;
	}
	myInterfaces[intf_num].pipes_attached=0;
}
