#include "globals.h"
#include <espconn.h>

#define IP_ANY	NULL												/// used with findPipe()
#define PORT_ANY 0													/// used with findPipe()
#define NUM_TRY	 400												/// number of try to send
#define UDP_TIME_TO_LIFE 60000										/// about 12 sec. for single UDP connection

LOCAL os_timer_t network_timer;

enum pipe_mode{
	NONE,
	TCP_SERVER,									// LS digit shows interface id(0/1), cond-se mode
	TCP_CLIENT,
	UDP_SERVER,
	UDP_CLIENT
};


typedef struct {
	s8 pipe_id;														// to which pipe attached
	u8 intf_num;													// number of interface pipe belongs to
	u16 try_count;
	enum pipe_mode	mode;											// see above
	union{
		esp_tcp tcpIP;
		esp_udp udpIP;
	};
	struct espconn pipe_con;
	struct espconn *con_p;											// pointer to current active espconn
} net_pipe;

LOCAL net_pipe *pipesWiFi[PIPE_COUNT]={NULL};					/// Every TCP pipe goes here


typedef struct {
	u8 Npipes;														// number of pipes
	u8 intf;
	union{
		esp_tcp tcpIP;
		esp_udp udpIP;
	};
	struct espconn pipe_con;
} server_pipe;

LOCAL server_pipe *pipesWiFiS[PIPE_COUNT]={NULL};						/// Every Server pipe goes here



LOCAL u8 MyIP[4]={0};												// router given current IP

void ICACHE_FLASH_ATTR wifi_close(u8);
void ICACHE_FLASH_ATTR network_check_ip(volatile u8*);
void ICACHE_FLASH_ATTR wifiRemovePipe(u8, u8);
void ICACHE_FLASH_ATTR pipeInit(u8);

/***********************************************************************************
 *  				HELP	F U N C T I O N S
 *
 ***********************************************************************************/

/******************************************************************************
 * FunctionName : compareIP
 * Description  : compare 2 espconn struct IP&Port
 * Parameters   : conn1 pointer to pipe conn, conn2 pointer to in_connection
 *
 * Returns      : TRUE if match, FALSE else
*******************************************************************************/
bool compareIP(struct espconn *conn1, struct espconn *conn2, bool remoteIP){
	u32 *my_ip1;
	u32 *my_ip2;
	u16 my_port1;
	u16 my_port2;
	if(conn1==NULL || conn2==NULL) return FALSE;
	if(remoteIP){
		if(conn1->type==ESPCONN_TCP){
			my_ip1=(u32*) conn1->proto.tcp->remote_ip;
			my_ip2=(u32*) conn2->proto.tcp->remote_ip;
			my_port1=conn1->proto.tcp->remote_port;
			my_port2=conn2->proto.tcp->remote_port;
		}
		else if(conn1->type==ESPCONN_UDP){
			my_ip1=(u32*) conn1->proto.udp->remote_ip;
			my_ip2=(u32*) conn2->proto.udp->remote_ip;
			my_port1=conn1->proto.udp->remote_port;
			my_port2=conn2->proto.udp->remote_port;
		}
		else return FALSE;
		//TRACE_DEBUG("R_ip: "IPSTR" R_pt:%d p_state:%d\r\n", IP2STR(conn1->proto.tcp->remote_ip), conn1->proto.tcp->remote_port, conn1->state);
	}
	else{
		if(conn1->type==ESPCONN_TCP){
			my_ip1=(u32*) conn1->proto.tcp->local_ip;
			my_ip2=(u32*) conn2->proto.tcp->local_ip;
			my_port1=conn1->proto.tcp->local_port;
			my_port2=conn2->proto.tcp->local_port;
		}
		else if(conn1->type==ESPCONN_UDP){
			my_ip1=(u32*) conn1->proto.udp->local_ip;
			my_ip2=(u32*) conn2->proto.udp->local_ip;
			my_port1=conn1->proto.udp->local_port;
			my_port2=conn2->proto.udp->local_port;
		}
		else return FALSE;
		//TRACE_DEBUG("L_ip: "IPSTR" L_pt:%d p_state:%d\r\n", IP2STR(conn1->proto.tcp->local_ip), conn1->proto.tcp->local_port, conn1->state);
	}
	if(*my_ip1==0 || *my_ip2==0){
		if(my_port1==my_port2) return TRUE;
			else return FALSE;
	}
	if(*my_ip1==*my_ip2 && my_port1==my_port2) return TRUE;
	else return FALSE;
}

/******************************************************************************
 * FunctionName : fillRxPipe(pipe)
 * Description  : compare 2 espconn struct IP&Port
 * Parameters   : conn1 pointer to pipe conn, conn2 pointer to in_connection
 *
 * Returns      : TRUE if match, FALSE else
*******************************************************************************/
void fillRxPipe(u8 pipe,  char *data, u16 len){
	u8 *s;
	u16 mylen=len;
	if((*myStreams[pipe].rxPipe.len +len)> (MAX_BUFFER)) mylen=MAX_BUFFER - *myStreams[pipe].rxPipe.len;
	s=myStreams[pipe].rxPipe.val+*myStreams[pipe].rxPipe.len;
	(*myStreams[pipe].rxPipe.len)+=mylen;
	fl_memcpy(s,data,mylen);
}
/***********************************************************************************
 *  				TCP		F U N C T I O N S
 *
 ***********************************************************************************/

/******************************************************************************
 * FunctionName : wifi_send
 * Description  : find a pipe attached with non-zero length of txPipe and transmit it trough wifi using tcp or udp
 * Parameters   : intf_num - number of interface. Not used for this processor
 *
 * Returns      :
*******************************************************************************/
void ICACHE_FLASH_ATTR wifi_send(u8 intf_num)
{
	u8 i;
	for (i=0; i<PIPE_COUNT; i++){
		if(pipesWiFi[i]>0){
			bool TCP_SEND=FALSE;
				//TRACE_DEBUG("PIPE%d STATE=%d\r\n",i,pipesWiFi[i]->con_p->state);

			 if(pipesWiFi[i]->con_p->state==ESPCONN_NONE || pipesWiFi[i]->con_p->state==ESPCONN_CLOSE){								// pipe reconnect
				pipeInit(i);
			 }
			 else if(pipesWiFi[i]->con_p->type==ESPCONN_TCP){
				if(pipesWiFi[i]->con_p->state==ESPCONN_WRITE){
					if(pipesWiFi[i]->try_count>0) {													// if a TCP_WRITE goes, wait to finish
						pipesWiFi[i]->try_count--;
						TRACE_DEBUG("P%d try=%d\r\n",i, pipesWiFi[i]->try_count);
						break;
					}
					else {
						espconn_disconnect(pipesWiFi[i]->con_p);
					}
				}
				else if(pipesWiFi[i]->con_p->state==ESPCONN_CONNECT || pipesWiFi[i]->con_p->state==ESPCONN_READ){							// send
					if(!TCP_SEND && *myStreams[pipesWiFi[i]->pipe_id].txPipe.len>0){
						u8 k=(pipesWiFi[i]->mode & 1) ? pipesWiFiS[pipesWiFi[i]->intf_num]->intf : pipesWiFi[i]->intf_num;
						if(pipesWiFi[i]->pipe_id>0 || (pipesWiFi[i]->pipe_id==0 && k==ReqPipe0.intf && i==ReqPipe0.i_pipe)){
						  TRACE_DEBUG("TSP Pipe%d send\r\n",i);
						  espconn_sent(pipesWiFi[i]->con_p, myStreams[pipesWiFi[i]->pipe_id].txPipe.val, *myStreams[pipesWiFi[i]->pipe_id].txPipe.len);
						  pipesWiFi[i]->try_count=NUM_TRY;
						  TCP_SEND=TRUE;
						}
					}
				}
			}
			else if(pipesWiFi[i]->con_p->type==ESPCONN_UDP && pipesWiFi[i]->pipe_con.state==ESPCONN_CONNECT){
				if(pipesWiFi[i]->mode==3){
					if(*myStreams[pipesWiFi[i]->pipe_id].txPipe.len>0){
						fl_memcpy(pipesWiFi[i]->con_p->proto.udp->remote_ip, pipesWiFi[i]->pipe_con.proto.udp->remote_ip, 4);
						pipesWiFi[i]->con_p->proto.udp->remote_port=pipesWiFi[i]->pipe_con.proto.udp->remote_port;
						espconn_sent(pipesWiFi[i]->con_p, myStreams[pipesWiFi[i]->pipe_id].txPipe.val, *myStreams[pipesWiFi[i]->pipe_id].txPipe.len);
						pipesWiFi[i]->pipe_con.state=ESPCONN_WRITE;
						TRACE_DEBUG("UDP Pipe%d send to %d.%d.%d.%d:%d\r\n",i,pipesWiFi[i]->con_p->proto.udp->remote_ip[0],pipesWiFi[i]->con_p->proto.udp->remote_ip[1],pipesWiFi[i]->con_p->proto.udp->remote_ip[2],pipesWiFi[i]->con_p->proto.udp->remote_ip[3],pipesWiFi[i]->con_p->proto.udp->remote_port);
					}
					else {													// if UDP Server
						if(pipesWiFi[i]->try_count>0) pipesWiFi[i]->try_count++;
						if(pipesWiFi[i]->try_count>UDP_TIME_TO_LIFE){
							pipesWiFi[i]->pipe_con.state=ESPCONN_LISTEN;
							pipesWiFi[i]->try_count=0;
							if(myStreams[pipesWiFi[i]->pipe_id].pipe_count==1) bitclr(&myPipes, pipesWiFi[i]->pipe_id);
							TRACE_DEBUG("UDP Pipe back to Listen\r\n");
						}
					}
				}
				else if(*myStreams[pipesWiFi[i]->pipe_id].txPipe.len>0){
					TRACE_DEBUG("UDP Pipe%d send to %d.%d.%d.%d:%d\r\n",i,pipesWiFi[i]->con_p->proto.udp->remote_ip[0],pipesWiFi[i]->con_p->proto.udp->remote_ip[1],pipesWiFi[i]->con_p->proto.udp->remote_ip[2],pipesWiFi[i]->con_p->proto.udp->remote_ip[3],pipesWiFi[i]->con_p->proto.udp->remote_port);
					espconn_sent(pipesWiFi[i]->con_p, myStreams[pipesWiFi[i]->pipe_id].txPipe.val, *myStreams[pipesWiFi[i]->pipe_id].txPipe.len);
					pipesWiFi[i]->pipe_con.state=ESPCONN_WRITE;
				}
			}
		}
	}
}

/******************************************************************************
 * FunctionName : tcp_sentCb
 * Description  : clear sent pipe txPipe
 * Parameters   :
 *
 * Returns      :
*******************************************************************************/
void ICACHE_FLASH_ATTR TCPsentCb(void *arg)
{
	struct espconn *in_con=(struct espconn *)arg;
	u8 i;
	for(i=0; i<PIPE_COUNT; i++){
		if(pipesWiFi[i]>0 && pipesWiFi[i]->con_p==in_con){
			*myStreams[pipesWiFi[i]->pipe_id].txPipe.len=0;
			pipesWiFi[i]->try_count=0;															// clear try
			TRACE_DEBUG("TCP%d try=%d\r\n",i, pipesWiFi[i]->try_count);
			break;
		}
	}
}

void ICACHE_FLASH_ATTR UDPsentCb(void *arg)
{
	struct espconn *in_con=(struct espconn *)arg;
	u8 i;
	for(i=0; i<PIPE_COUNT; i++){
		if(pipesWiFi[i]>0 && pipesWiFi[i]->con_p==in_con && (pipesWiFi[i]->mode!=3 || (pipesWiFi[i]->mode==3 && compareIP(&pipesWiFi[i]->pipe_con, in_con, 1)))){
			*myStreams[pipesWiFi[i]->pipe_id].txPipe.len=0;
			pipesWiFi[i]->try_count=1;															// clear try
			pipesWiFi[i]->pipe_con.state=ESPCONN_CONNECT;
			TRACE_DEBUG("UDP%d try=%d\r\n",i, pipesWiFi[i]->try_count);
			break;
		}
	}
}
/******************************************************************************
 * FunctionName : rcvCb	- receive callback
 * Description  : write tcp data to corresponding rxPipe
 * Parameters   : char *arg -espcon struct, data - point to data received, len - length of data received
 *
 * Returns      :
*******************************************************************************/
static void ICACHE_FLASH_ATTR rcvCb(void *arg, char *data, u16 len)
{
	struct espconn *in_con=(struct espconn *)arg;
	u8 i;
	s8 doRock=-1;
		TRACE_DEBUG("data in P");
	for(i=0; i<PIPE_COUNT; i++){
		if(pipesWiFi[i]==0) continue;
		if(pipesWiFi[i]->con_p==in_con){
			TRACE_DEBUG("%d %d\r\n", pipesWiFi[i]->pipe_id, in_con);
			if(pipesWiFi[i]->pipe_id==0) doRock=pipesWiFi[i]->pipe_id;
			fillRxPipe(pipesWiFi[i]->pipe_id, data, len);
		}
	}
	if(doRock>=0){																				// if Console
		u8 k=(pipesWiFi[doRock]->mode & 1) ? pipesWiFiS[pipesWiFi[doRock]->intf_num]->intf : pipesWiFi[doRock]->intf_num;
		TRACE_DEBUG("ROCK ");
		if(rock(k)){
			ReqPipe0.intf=k;
			ReqPipe0.i_pipe=doRock;
			*myStreams[0].rxPipe.len = 0;
			TRACE_DEBUG("P0->%d, intf->%d\r\n", ReqPipe0.i_pipe, ReqPipe0.intf);
		}
	}
}
/******************************************************************************
 * FunctionName : UDPSrcvCb	- receive callback
 * Description  : If first connection fill a LISTEN pipe with IP:port. Write data to corresponding rxPipe
 * Parameters   : char *arg -espcon struct, data - point to data received, len - length of data received
 *
 * Returns      :
*******************************************************************************/
static void ICACHE_FLASH_ATTR UDPSrcvCb(void *arg, char *data, u16 len)
{
	struct espconn *in_con=(struct espconn *)arg;
	u8 i,j;
	u16 old_try=0;
	s8 list_pipe=-1, oldest_pipe=-1, doRock=-1;
		TRACE_DEBUG("UDPSdata in P %d", in_con);

	for(j=0; j<PIPE_COUNT; j++){
		if(pipesWiFiS[j]==0) continue;
		if(&pipesWiFiS[j]->pipe_con==in_con){														// find corresponding Server pipe
			for(i=0;i<PIPE_COUNT; i++){
				if(pipesWiFi[i]==0 || pipesWiFi[i]->pipe_con.type!=ESPCONN_UDP) continue;
				if(pipesWiFi[i]->intf_num==j){
					if(compareIP(&pipesWiFi[i]->pipe_con, in_con, 1)) break;	// if remote IP passes
					if(pipesWiFi[i]->pipe_con.state==ESPCONN_LISTEN && list_pipe<0) list_pipe=i;
					if(pipesWiFi[i]->pipe_con.state==ESPCONN_CONNECT && pipesWiFi[i]->try_count > old_try){
						old_try=pipesWiFi[i]->try_count;
						oldest_pipe=i;
					}
				}
			}
			if(i==PIPE_COUNT){
				if(list_pipe>=0) i=list_pipe;
				else if(oldest_pipe>=0) i=oldest_pipe;
				else return;
				fl_memcpy(pipesWiFi[i]->pipe_con.proto.udp->remote_ip, in_con->proto.udp->remote_ip, 4);
				pipesWiFi[i]->pipe_con.proto.udp->remote_port = in_con->proto.udp->remote_port;
				pipesWiFi[i]->con_p=in_con;
				pipesWiFi[i]->pipe_con.state=ESPCONN_CONNECT;
			}
			if(pipesWiFi[i]->pipe_id==0) doRock=pipesWiFi[i]->pipe_id;
			fillRxPipe(pipesWiFi[i]->pipe_id, data, len);
			pipesWiFi[i]->try_count=1;
			bitset(&myPipes,pipesWiFi[i]->pipe_id);
			 TRACE_DEBUG("UDP use P%d\r\n",i);
		}
	}
	if(doRock>=0){																				// if Console
		u8 k=(pipesWiFi[doRock]->mode & 1) ? pipesWiFiS[pipesWiFi[doRock]->intf_num]->intf : pipesWiFi[doRock]->intf_num;
		TRACE_DEBUG("ROCK ");
		if(rock(k)){
			ReqPipe0.intf=k;
			ReqPipe0.i_pipe=doRock;
			*myStreams[0].rxPipe.len = 0;
		}
	}
}
//---------------------------------------------------------------------------------
static void ICACHE_FLASH_ATTR tcpReconCb(void *arg, sint8 err){
	struct espconn *in_con=(struct espconn *) arg;
	u8 i;
	TRACE_DEBUG("Reconn ");
	for (i=0;i<PIPE_COUNT; i++){
		if(pipesWiFi[i]>0 && pipesWiFi[i]->con_p==in_con){
			TRACE_DEBUG("pipe %d, id %d, state %d", i, pipesWiFi[i]->pipe_id, in_con->state);
			if(pipesWiFi[i]->pipe_id==-1){
				espconn_delete(pipesWiFi[i]->con_p);							// if called from closePipe
				my_free((char*) pipesWiFi[i]);
				pipesWiFi[i]=0;
				TRACE_DEBUG("Pipe %d free\r\n", i)
			}
			else if(myStreams[pipesWiFi[i]->pipe_id].pipe_count==1) bitclr(&myPipes, pipesWiFi[i]->pipe_id);
			break;
		}
		else if(pipesWiFiS[i]>0 && &pipesWiFiS[i]->pipe_con==in_con){
			//pipesWiFiS[i]->Npipes--;
			if(pipesWiFiS[i]->Npipes<=0) {
				espconn_delete(&pipesWiFiS[i]->pipe_con);							// according to >9.5
				my_free((char*) pipesWiFiS[i]);
				pipesWiFiS[i]=0;
				TRACE_DEBUG("Server %d free\r\n", i);
			}
			else espconn_tcp_set_max_con_allow(&pipesWiFiS[i]->pipe_con,pipesWiFiS[i]->Npipes);
			break;
		}
	}
	TRACE_DEBUG("!\r\n");
}

//---------------------------------------------------------------------------------
static void ICACHE_FLASH_ATTR tcpDisconCb(void *arg){
	struct espconn *in_con=(struct espconn *) arg;
	u8 i;
	TRACE_DEBUG("P_disconn state %d, %d\r\n", in_con->state, in_con);
	for (i=0;i<PIPE_COUNT; i++){
		if(pipesWiFi[i]>0 && pipesWiFi[i]->con_p==in_con){
			TRACE_DEBUG("pipe %d, id %d, state %d", i, pipesWiFi[i]->pipe_id, in_con->state);
			if(pipesWiFi[i]->pipe_id==-1){
				espconn_delete(pipesWiFi[i]->con_p);							// if called from closePipe
				my_free((char*) pipesWiFi[i]);
				pipesWiFi[i]=0;
				TRACE_DEBUG("Pipe %d free\r\n", i)
			}
			else if(myStreams[pipesWiFi[i]->pipe_id].pipe_count==1) bitclr(&myPipes, pipesWiFi[i]->pipe_id);
			break;
		}
		else if(pipesWiFiS[i]>0 && &pipesWiFiS[i]->pipe_con==in_con){
			//pipesWiFiS[i]->Npipes--;
			if(pipesWiFiS[i]->Npipes<=0) {
				espconn_delete(&pipesWiFiS[i]->pipe_con);							// according to >9.5
				my_free((char*) pipesWiFiS[i]);
				pipesWiFiS[i]=0;
				TRACE_DEBUG("Server %d free\r\n", i);
			}
			else espconn_tcp_set_max_con_allow(&pipesWiFiS[i]->pipe_con,pipesWiFiS[i]->Npipes);
			break;
		}
	}
}

//---------------------------------------------------------------------------------
static void ICACHE_FLASH_ATTR tcpConnectCb(void *arg){
	struct espconn *newTCP=(struct espconn *) arg;
	u8 i;

	// find the index of Pipe in pipesAttachdTCP[array]
	for(i=0; i<PIPE_COUNT; i++){
		if(pipesWiFi[i]>0 && pipesWiFi[i]->con_p==newTCP){
			espconn_regist_recvcb(pipesWiFi[i]->con_p, rcvCb);
			espconn_regist_sentcb(pipesWiFi[i]->con_p, TCPsentCb);
			TRACE_DEBUG("P%d conn\r\n", pipesWiFi[i]->pipe_id);
			bitset(&myPipes,pipesWiFi[i]->pipe_id);
			break;
		}
	}
}

static void ICACHE_FLASH_ATTR tcpServerConnectCb(void *arg){
	struct espconn *newTCP=(struct espconn *) arg;
	u8 i;
	// find the index of a server Pipe in pipesAttachdWiFi[array]
	for(i=0; i<PIPE_COUNT; i++){
		if(pipesWiFi[i]>0 && pipesWiFi[i]->mode==1 && pipesWiFi[i]->con_p->state==ESPCONN_LISTEN && compareIP(pipesWiFi[i]->con_p, newTCP, 0)){			// if Server pipe and waiting
			pipesWiFi[i]->con_p=newTCP;																// assign to it
		    espconn_regist_disconcb(newTCP, tcpDisconCb);       // Register disconnect callback
		    espconn_regist_reconcb(newTCP, tcpReconCb);         // Register reconnection function
		    espconn_regist_sentcb(newTCP, TCPsentCb);
			espconn_regist_recvcb(newTCP, rcvCb);
			espconn_regist_time(newTCP, 360, 1);				// keep connection 6 min=360 sec
				TRACE_DEBUG("P%d attached,state=%d, arg=%d\r\n",pipesWiFi[i]->pipe_id,pipesWiFi[i]->con_p->state,newTCP);
			bitset(&myPipes, pipesWiFi[i]->pipe_id);
			return;
		}
	}
    espconn_regist_disconcb(newTCP, tcpDisconCb);       // Register disconnect callback
	espconn_disconnect(newTCP);									// if no pipe dedicated
	TRACE_DEBUG("Sdiscon %d\r\n",newTCP);
}

/***********************************************************************************
 *  				P I P E 		F U N C T I O N S
 *
 ***********************************************************************************/

/******************************************************************************
 * FunctionName : wifiAddPipe
 * Description  : attach a pipe to interface
 * Parameters   : u8 pipe, u8 intf_number
 *
 * Returns      :
*******************************************************************************/
void ICACHE_FLASH_ATTR wifiAddPipe(u8 pipe, u8 intf_num, u8 *params){		// P<pipeN>{interfaceN, parems= S(erver),"TCP","0.0.0.0",port, num_connections} or P<pipeN>{interfaceN, params= C(lient), "TCP","192.168.1.3",port}
	u8 myT;
	u8 IP0=0, IP1=0, IP2=0, IP3=0, *myp=params;
	u16 myPort=0;
	u8 i=0, num_conn=0;
	s8 new_pipe=-1;

	const char *HELP="{interfaceN, S/C Server/Client,\"TCP\"/\"UDP\",\"IP.IP.IP.IP\",port}";

	if(find_char(params, '?')) {toP0((u8*) HELP,0,0); return;}
	if(myInterfaces[intf_num].state==INTF_CLOSE) return;
	myT=(*params=='S') ? 1:2;											// 0=NONE, 1=Server 2=Client
	while(*myp !=34) myp++;													// point to " of "TCP"/"UDP"
	myp++;
	if(*myp!='T') myT+=2;													// NONE/TCP_SERVER/TCP_CLIENT/UDP_SERVER/UDP_CLIENT
	while(*myp !=',') myp++;												// goto IP
	myp++;
	//if(myT==1) goto L_1;													// if Server jump to port try to comment and use IP to spacify what interface to run server
	if(*myp==34){															// if "IP" follows
		u8 *p=out$.val;
		Ex(&myp,0);
		out$.val[out$.len]=0;
		IP0= indexE(&p,10);
		p++;
		IP1= indexE(&p,10);
		p++;
		IP2= indexE(&p,10);
		p++;
		IP3= indexE(&p,10);
		if(*myp==34) myp++;
	}
L_1:
	myPort=indexE(&myp,10);
	//if(myT==1 || myT==3) num_conn=indexE(&myp, 10);
	//if(num_conn==0) num_conn=1;
		TRACE_DEBUG("myT=%d,%d.%d.%d.%d:%d,n=%d\r\n", myT,IP0,IP1,IP2,IP3,myPort,num_conn);
	//---------------- All the parameters read so find one empty connection ----------------
	for(i=0; i<PIPE_COUNT; i++){
		if(pipesWiFi[i]>0 && pipesWiFi[i]->pipe_id==pipe){
			wifiRemovePipe(pipe, ((pipesWiFi[i]->mode) & 1) ? pipesWiFiS[pipesWiFi[i]->intf_num]->intf : pipesWiFi[i]->intf_num);
			while(pipesWiFi[i]>0);																// wait to close same pipe
		}
		if(pipesWiFi[i]==0 && new_pipe==-1) new_pipe=i;
	}
	if(new_pipe==-1) {toP0("No free WiFi Pipe",0,0); return;}										// if no free pipe
	i=new_pipe;
	createPipe(pipe);
	if(myStreams[pipe].pipe_count==1) bitclr(&myPipes, pipe);
	pipesWiFi[i]=(net_pipe*) my_malloc(sizeof(net_pipe));
	if(pipesWiFi[i]==0) {toP0("Out of mem for WiFi Pipe",0,0); return;}
    pipesWiFi[i]->pipe_id=pipe;
    pipesWiFi[i]->mode=myT;																			//myT = 1..4, or 0x11..0x14 depend on interface attached;
    pipesWiFi[i]->intf_num=intf_num;																// in case Client pipe
    myInterfaces[intf_num].pipes_attached++;
    pipesWiFi[i]->con_p=&pipesWiFi[i]->pipe_con;													// set pointer to own espconn so we can use as temp set. After connecting a real tcp, it will be replaced in tcpServerConnectCb()
        TRACE_DEBUG("myT=%d, pipe_id=%d\r\n", myT,pipesWiFi[i]->pipe_id);
	switch (myT){
		case 1:{																					// TCP Server
			u8 j;
			s8 empty_Spipe=-1;
			pipesWiFi[i]->pipe_con.type=ESPCONN_TCP;
			pipesWiFi[i]->pipe_con.proto.tcp=&pipesWiFi[i]->tcpIP;									// set pointer to IP vars
			pipesWiFi[i]->pipe_con.state=ESPCONN_NONE;												// to initPipe later
			pipesWiFi[i]->tcpIP.local_port=myPort;
			pipesWiFi[i]->tcpIP.local_ip[0]=IP0;
			pipesWiFi[i]->tcpIP.local_ip[1]=IP1;
			pipesWiFi[i]->tcpIP.local_ip[2]=IP2;
			pipesWiFi[i]->tcpIP.local_ip[3]=IP3;
			// Find an empty Server pipe and create server
			for(j=0;j<PIPE_COUNT;j++){
				if(pipesWiFiS[j]==0){
					if(empty_Spipe==-1) empty_Spipe=j;
					continue;
				}
				if(pipesWiFiS[j]->pipe_con.type==ESPCONN_TCP && compareIP(pipesWiFi[i]->con_p, &pipesWiFiS[j]->pipe_con, 0)){									// check if same server
					pipesWiFi[i]->pipe_con.state=ESPCONN_LISTEN;									// not to init later
					pipesWiFi[i]->intf_num=j;
					pipesWiFiS[j]->Npipes++;
					TRACE_DEBUG("TCPServer the Same\r\n");
					break;
				}
			}
			if(j==PIPE_COUNT){
				if(empty_Spipe==-1){toP0("No free Server Pipe",0,0); return;}							// if no free pipe
				j=empty_Spipe;
				pipesWiFiS[j]=(server_pipe*) my_malloc(sizeof(server_pipe));
				if(pipesWiFiS[j]==0) {toP0("Out of mem for Server Pipe",0,0); return;}
				pipesWiFi[i]->intf_num=j;																// set pipe intf_num to hold Server index
				// Server init
				pipesWiFiS[j]->Npipes=1;
				pipesWiFiS[j]->intf=intf_num;															// interface will be here
				pipesWiFiS[j]->pipe_con.type=ESPCONN_TCP;                                    			// We want to make a TCP connection
				pipesWiFiS[j]->pipe_con.state=ESPCONN_NONE;                                 		 	// Set default state to none
				pipesWiFiS[j]->pipe_con.proto.tcp=&pipesWiFiS[j]->tcpIP;                  				// Give a pointer to last one TCP var
				pipesWiFiS[j]->pipe_con.proto.tcp->local_port=myPort;                         			// Set Server port
				pipesWiFiS[j]->pipe_con.proto.tcp->local_ip[0]=IP0;                       				// set IP
				pipesWiFiS[j]->pipe_con.proto.tcp->local_ip[1]=IP1;
				pipesWiFiS[j]->pipe_con.proto.tcp->local_ip[2]=IP2;
				pipesWiFiS[j]->pipe_con.proto.tcp->local_ip[3]=IP3;
				espconn_regist_connectcb(&pipesWiFiS[j]->pipe_con, tcpServerConnectCb);   				// Register connect callback
				myInterfaces[intf_num].state=INTF_CONNECT;
					TRACE_DEBUG("TCPServer Init\r\n");
			}
			espconn_tcp_set_max_con_allow(&pipesWiFiS[j]->pipe_con, pipesWiFiS[j]->Npipes);
		}
		break;
		case 2:{																	// TCP Client myT=2
			pipesWiFi[i]->pipe_con.type=ESPCONN_TCP;
			pipesWiFi[i]->pipe_con.proto.tcp=&pipesWiFi[i]->tcpIP;						// set pointer to IP vars
			pipesWiFi[i]->pipe_con.state=ESPCONN_NONE;
			//my_conn[i].pipe_con.link_cnt=pipe;									// link_cnt=pipe# - works only for client not for server!
			pipesWiFi[i]->pipe_con.proto.tcp->remote_ip[0]=IP0;
			pipesWiFi[i]->pipe_con.proto.tcp->remote_ip[1]=IP1;
			pipesWiFi[i]->pipe_con.proto.tcp->remote_ip[2]=IP2;
			pipesWiFi[i]->pipe_con.proto.tcp->remote_ip[3]=IP3;
			pipesWiFi[i]->pipe_con.proto.tcp->remote_port=myPort;
			pipesWiFi[i]->pipe_con.proto.tcp->local_port=espconn_port();           	// Ask a free local port to the API

			espconn_regist_connectcb(&pipesWiFi[i]->pipe_con, tcpConnectCb);  			// Register Client ST/AP connect callback
			espconn_regist_reconcb(&pipesWiFi[i]->pipe_con, tcpReconCb);      			// Register Client ST/AP reconnection function
			espconn_regist_disconcb(&pipesWiFi[i]->pipe_con, tcpDisconCb);    			// Register Client ST/AP disconnect callback
			myInterfaces[intf_num].state=INTF_CONNECT;
		}
		break;
		case 3:{																	// UDP Server
			u8 j;
			s8 empty_Spipe=-1;
			pipesWiFi[i]->pipe_con.type=ESPCONN_UDP;
			pipesWiFi[i]->pipe_con.proto.udp=&pipesWiFi[i]->udpIP;						// IP vars
			pipesWiFi[i]->pipe_con.state=ESPCONN_NONE;
			pipesWiFi[i]->udpIP.local_port=myPort;
			pipesWiFi[i]->udpIP.local_ip[0]=IP0;
			pipesWiFi[i]->udpIP.local_ip[1]=IP1;
			pipesWiFi[i]->udpIP.local_ip[2]=IP2;
			pipesWiFi[i]->udpIP.local_ip[3]=IP3;
			//espconn_regist_recvcb(&pipesWiFiS[j]->pipe_con, UDPSrcvCb);   				// Register connect callback
			espconn_regist_sentcb(pipesWiFi[i]->con_p, UDPsentCb);
			// Find an empty Server pipe and create server
			for(j=0;j<PIPE_COUNT;j++){
				if(pipesWiFiS[j]==0){
					if(empty_Spipe==-1) empty_Spipe=j;
					continue;
				}
				if(pipesWiFiS[j]->pipe_con.type==ESPCONN_UDP && compareIP(pipesWiFi[i]->con_p, &pipesWiFiS[j]->pipe_con, 0)){									// check if same server
					//pipesWiFi[i]->pipe_con.state=ESPCONN_LISTEN;									// not to init later
					pipesWiFi[i]->intf_num=j;
					pipesWiFiS[j]->Npipes++;
					TRACE_DEBUG("UDPServer the Same\r\n");
					break;
				}
			}
			if(j==PIPE_COUNT){																			// a new Server we need
				if(empty_Spipe==-1){toP0("No free Server Pipe",0,0); return;}							// if no free pipe
				j=empty_Spipe;
				pipesWiFiS[j]=(server_pipe*) my_malloc(sizeof(server_pipe));
				if(pipesWiFiS[j]==0) {toP0("Out of mem for Server Pipe",0,0); return;}
				pipesWiFi[i]->intf_num=j;																// set pipe intf_num to hold Server index
				// Server init
				pipesWiFiS[j]->Npipes=1;
				pipesWiFiS[j]->intf=intf_num;															// 1-St, 2-AP
				pipesWiFiS[j]->pipe_con.type=ESPCONN_UDP;                                    			// We want to make a TCP connection
				pipesWiFiS[j]->pipe_con.state=ESPCONN_NONE;                                 		 		// Set default state to none
				pipesWiFiS[j]->pipe_con.proto.udp=&pipesWiFiS[j]->udpIP;                  					// Give a pointer to last one TCP var
				pipesWiFiS[j]->pipe_con.proto.udp->local_port=myPort;                         			// Set Server port
				pipesWiFiS[j]->pipe_con.proto.udp->local_ip[0]=IP0;                       				// set IP
				pipesWiFiS[j]->pipe_con.proto.udp->local_ip[1]=IP1;
				pipesWiFiS[j]->pipe_con.proto.udp->local_ip[2]=IP2;
				pipesWiFiS[j]->pipe_con.proto.udp->local_ip[3]=IP3;
				espconn_regist_recvcb(&pipesWiFiS[j]->pipe_con, UDPSrcvCb);   				// Register connect callback
				espconn_regist_sentcb(&pipesWiFiS[j]->pipe_con, UDPsentCb);
				myInterfaces[intf_num].state=INTF_CONNECT;
					TRACE_DEBUG("UDPServer Init %d\r\n", &pipesWiFi[j]->pipe_con);
			}
		}
		break;
		case 4:{ 																	// UDP Client
			pipesWiFi[i]->pipe_con.type=ESPCONN_UDP;
			pipesWiFi[i]->pipe_con.proto.udp=&pipesWiFi[i]->udpIP;						// IP vars
			pipesWiFi[i]->pipe_con.state=ESPCONN_NONE;
			pipesWiFi[i]->pipe_con.proto.udp->remote_ip[0]=IP0;
			pipesWiFi[i]->pipe_con.proto.udp->remote_ip[1]=IP1;
			pipesWiFi[i]->pipe_con.proto.udp->remote_ip[2]=IP2;
			pipesWiFi[i]->pipe_con.proto.udp->remote_ip[3]=IP3;
			pipesWiFi[i]->pipe_con.proto.udp->remote_port=myPort;
			pipesWiFi[i]->pipe_con.proto.udp->local_port=espconn_port();  				// Ask a free local port to the API
			myInterfaces[intf_num].state=INTF_CONNECT;
			espconn_regist_sentcb(pipesWiFi[i]->con_p, UDPsentCb);
			 //TRACE_DEBUG("state=%d\r\n", pipesWiFi[i]->pipe_con.state);
			//espconn_sent(pipesWiFi[i]->con_p,"TS\r\n",sizeof("TS\r\n"));
		}
		break;
	}
	TRACE_DEBUG("Pipe=%d, state=%d\r\n", pipesWiFi[i]->pipe_id, pipesWiFi[i]->pipe_con.state);
}
/******************************************************************************
 * FunctionName : wifiRemovePipe
 * Description  : attach a pipe to interface
 * Parameters   : u8 pipe, u8 intf_number
 *
 * Returns      :
*******************************************************************************/
void ICACHE_FLASH_ATTR wifiRemovePipe(u8 pipe, u8 intf_num){
	u8 i;
	// find the index of Pipe in pipesAttachdTCP[array]
	for(i=0; i<PIPE_COUNT; i++){
		if(pipesWiFi[i]>0 && pipesWiFi[i]->pipe_id==pipe){
			pipesWiFi[i]->pipe_id=-1;												// first remove from send list
				TRACE_DEBUG("Rem P%d index=%d mode=%d\r\n",pipe,i,pipesWiFi[i]->mode);
			switch(pipesWiFi[i]->mode){
				case 1:{																	// case TCP Server Pipe
					pipesWiFiS[pipesWiFi[i]->intf_num]->Npipes--;
					if(pipesWiFi[i]->con_p->state==ESPCONN_CONNECT) espconn_disconnect(pipesWiFi[i]->con_p);		// Server's check for closure will be done in ClosePipeCb
					else{
						if(pipesWiFiS[pipesWiFi[i]->intf_num]->Npipes<=0) {
							espconn_delete(&pipesWiFiS[pipesWiFi[i]->intf_num]->pipe_con);							// according to >9.5
							my_free((char*) pipesWiFiS[pipesWiFi[i]->intf_num]);
							pipesWiFiS[pipesWiFi[i]->intf_num]=0;
							TRACE_DEBUG("Server %d free\r\n", pipesWiFi[i]->intf_num);
						}
						else espconn_tcp_set_max_con_allow(&pipesWiFiS[pipesWiFi[i]->intf_num]->pipe_con,pipesWiFiS[pipesWiFi[i]->intf_num]->Npipes);
					}
						espconn_delete(pipesWiFi[i]->con_p);							// if called from closePipe
						my_free((char*) pipesWiFi[i]);
						pipesWiFi[i]=0;
						TRACE_DEBUG("Pipe %d free\r\n", i);

				}
				break;
				case 2:{																						// case TCP Client Pipe
					if(pipesWiFi[i]->con_p->state==ESPCONN_CONNECT) espconn_disconnect(pipesWiFi[i]->con_p);
					else{
						espconn_delete(pipesWiFi[i]->con_p);													// if called from closePipe
						my_free((char*) pipesWiFi[i]);
						pipesWiFi[i]=0;
						TRACE_DEBUG("Pipe %d free\r\n", i)
					}
				}
				break;
				case 3: {
					pipesWiFiS[pipesWiFi[i]->intf_num]->Npipes--;
					if(pipesWiFiS[pipesWiFi[i]->intf_num]->Npipes<=0) {
						espconn_delete(&pipesWiFiS[pipesWiFi[i]->intf_num]->pipe_con);							// according to >9.5
						my_free((char*) pipesWiFiS[pipesWiFi[i]->intf_num]);
						pipesWiFiS[pipesWiFi[i]->intf_num]=0;
					}
					espconn_delete(pipesWiFi[i]->con_p);								// case UDP Server Pipe
					my_free((char*) pipesWiFi[i]);
					pipesWiFi[i]=0;
				}
				break;
				case 4:{																	// case UDP Client Pipe
					espconn_delete(pipesWiFi[i]->con_p);
					my_free((char*) pipesWiFi[i]);
					pipesWiFi[i]=0;
				}
				break;
			}
			/*
			pipesWiFi[i]->con_p=&pipesWiFi[i]->pipe_con;							// restore pointer if server connection else simply rewrite
			pipesWiFi[i]->pipe_con.state=ESPCONN_NONE;
			pipesWiFi[i]->mode=NONE;
			*/
			freePipeCheck(pipe);
			if(--(myInterfaces[intf_num].pipes_attached)<=0){
				myInterfaces[intf_num].pipes_attached=0;
				myInterfaces[intf_num].state=INTF_OPEN;
			}
			break;
		}
	}
}
/******************************************************************************
 * FunctionName : pipeInit
 * Description  : start a initialised pipe
 * Parameters   : u8 pipe, u8 intf_number
 *
 * Returns      :
*******************************************************************************/
void ICACHE_FLASH_ATTR pipeInit(u8 j)
{
	u8 k;
	TRACE_DEBUG("Init P%d, mode=%d, state=%d\r\n",j, pipesWiFi[j]->mode,pipesWiFi[j]->pipe_con.state);
	k=(pipesWiFi[j]->mode & 1) ? pipesWiFiS[pipesWiFi[j]->intf_num]->intf : pipesWiFi[j]->intf_num;
	if(myInterfaces[k].subtype==1){																				// if Station
		network_check_ip(&myInterfaces[k].state);
		if(myInterfaces[k].state < INTF_OPEN) return;					// back if not ready
	}
	switch (pipesWiFi[j]->mode){
		case 1:{																		// TCP Server
			if(pipesWiFiS[pipesWiFi[j]->intf_num]->pipe_con.state==ESPCONN_NONE){
				espconn_accept(&pipesWiFiS[pipesWiFi[j]->intf_num]->pipe_con);
					TRACE_DEBUG("TCPServer Start\r\n");

			}
			else{
				TRACE_DEBUG("Pipe%d Restore\r\n",j);
				pipesWiFi[j]->con_p=&pipesWiFi[j]->pipe_con;					// restore pointer if server connection else simply rewrite
				if(myStreams[pipesWiFi[j]->pipe_id].pipe_count==1) bitclr(&myPipes, pipesWiFi[j]->pipe_id);		// if pipe disconnected & just one interface attached, clear Pipe bit(Pipe nor ready)
			}
			pipesWiFi[j]->pipe_con.state=ESPCONN_LISTEN;
		}
		break;
		case 2:{																		// TCP Client
			pipesWiFi[j]->pipe_con.state=ESPCONN_NONE;
			espconn_connect(pipesWiFi[j]->con_p);
				TRACE_DEBUG("try to conn %d,\r\n", j);
		}
		break;
		case 3:{																		// UDP Server
			if(pipesWiFiS[pipesWiFi[j]->intf_num]->pipe_con.state==ESPCONN_NONE){
				espconn_create(&pipesWiFiS[pipesWiFi[j]->intf_num]->pipe_con);
				pipesWiFiS[pipesWiFi[j]->intf_num]->pipe_con.state=ESPCONN_LISTEN;
				TRACE_DEBUG("UDPServer Start\r\n");
			}
			pipesWiFi[j]->con_p->state=ESPCONN_LISTEN;
			TRACE_DEBUG("UDPServer Pipe %d Start\r\n",j);
		}
		break;
		case 4:{																		// UDP Client
			if(espconn_create(pipesWiFi[j]->con_p)==0) bitset(&myPipes, pipesWiFi[j]->pipe_id);
			espconn_regist_recvcb(pipesWiFi[j]->con_p, rcvCb);   				// Register connect callback
			pipesWiFi[j]->con_p->state=ESPCONN_CONNECT;
			TRACE_DEBUG("UDP conn %d,\r\n", j);
		}
		break;
	}
}
//---------------------------------------------------------------------------------
/***********************************************************************************
 *  				N E T W O R K 		F U N C T I O N S
 *
 ***********************************************************************************/
void ICACHE_FLASH_ATTR network_check_ip(volatile u8* state)
{
    struct ip_info ipconfig;
    u8 status;
    os_timer_disarm(&network_timer);                // Disarm timer
    wifi_get_ip_info(STATION_IF, &ipconfig);        // Get Wifi info
    status=wifi_station_get_connect_status();
    if ( status == STATION_GOT_IP && ipconfig.ip.addr != 0)
    {
    	MyIP[0]=(u8) (ipconfig.ip.addr>>24);
    	MyIP[1]=(u8) (ipconfig.ip.addr>>16);
    	MyIP[2]=(u8) (ipconfig.ip.addr>>8);
    	MyIP[3]=(u8) ipconfig.ip.addr;
        //network_start();                            // will be done in wifiSend
        if(*state==INTF_CREATE){
        	u8 i,k;
        	*state=INTF_OPEN;
        	for(i=0;i<PIPE_COUNT; i++){
        		if(pipesWiFi[i]>0){
        			k=(pipesWiFi[i]->mode & 1) ? pipesWiFiS[pipesWiFi[i]->intf_num]->intf : pipesWiFi[i]->intf_num;
        			if(myInterfaces[k].subtype==1){
        				*state=INTF_CONNECT;
        				break;
        			}
         		}
        	}
        	TRACE_DEBUG("TCPgotIP=%d.%d.%d.%d\r\n",MyIP[3],MyIP[2],MyIP[1],MyIP[0]);
        }

    }
    else
    {
        	TRACE_DEBUG("Status=%d\n\r", status);
    	if(status==0) wifi_station_connect();
    	if(status>1){
    		u8 i,k;
    		for(i=0; i<PIPE_COUNT;i++){
    			if(pipesWiFiS[i]>0 && myInterfaces[pipesWiFiS[i]->intf].subtype==1){
    				pipesWiFiS[i]->pipe_con.state=ESPCONN_NONE;
    			}
    			if(pipesWiFi[i]>0){
    				k=(pipesWiFi[i]->mode & 1) ? pipesWiFiS[pipesWiFi[i]->intf_num]->intf : pipesWiFi[i]->intf_num;
    				if(myInterfaces[k].subtype==1){									// if Server pipe && Station
						pipesWiFi[i]->con_p=&pipesWiFi[i]->pipe_con;
						pipesWiFi[i]->pipe_con.state=ESPCONN_CLOSE;
						if(myStreams[pipesWiFi[i]->pipe_id].pipe_count==1) bitclr(&myPipes, pipesWiFi[i]->pipe_id);
    				}
    			}
    		}
    	}
        os_timer_setfn(&network_timer, (os_timer_func_t *)network_check_ip, state);
        os_timer_arm(&network_timer, 1000, 0);
        *state=INTF_CREATE;
    }
}
//---------------------------------------------------------------------------------
// network init function
void ICACHE_FLASH_ATTR network_init(volatile u8* state)
{
    os_timer_disarm(&network_timer);
    os_timer_setfn(&network_timer, (os_timer_func_t *)network_check_ip, state);
    os_timer_arm(&network_timer, 1000, 0);
}

/***********************************************************************************
 *  				I N T E R F A C E 		F U N C T I O N S
 *
 ***********************************************************************************/

/******************************************************************************
 * FunctionName : wifi_close
 * Description  : remove all the pipes attached and close wifi
 * Parameters   : u8 intf_number
 *
 * Returns      :
*******************************************************************************/
void ICACHE_FLASH_ATTR wifi_close(u8 intf_num){
	u8 j,k;
	for (j=0;j<PIPE_COUNT;j++){
		if(pipesWiFi[j]==0) continue;
		k=(pipesWiFi[j]->mode & 1) ? pipesWiFiS[pipesWiFi[j]->intf_num]->intf : pipesWiFi[j]->intf_num;
		if(k==intf_num){															// check if pipe belongs to interface
			wifiRemovePipe(pipesWiFi[j]->pipe_id, intf_num);
		}
	}
	if(myInterfaces[intf_num].subtype==1){
		if(wifi_station_dhcpc_status()==DHCP_STARTED) wifi_station_dhcpc_stop();
		if(wifi_station_get_connect_status()>STATION_IDLE) wifi_station_disconnect();
		//wifi_set_opmode(wifi_get_opmode() & ~1);
	}
	else{
		if(wifi_softap_dhcps_status()==DHCP_STARTED) wifi_softap_dhcps_stop();
		wifi_set_opmode(wifi_get_opmode() & ~2);
	}
	myInterfaces[intf_num].state=INTF_CLOSE;
}
/******************************************************************************
 * FunctionName : initWiFi(intf_number, params)
 * Description  : remove all the pipes attached and close wifi
 * Parameters   : u8 intf_number, string (parameters)={mode=1=Station, "ssid", "pw", (optional)"IP.IP.IP.IP","GW.GW.GW.GW","NM.NM.NM.NM"} or {mode=2=AP, "ssid", "pw", "IP.IP.IP.IP", auth_mode, channel, use_dhcp=1/0}
 *
 * Returns      :
*******************************************************************************/
void  ICACHE_FLASH_ATTR initWiFi(u8 intf_num, u8 *params){
	#define defaultIP	IP4_ADDR(&myIP.ip,192,168,4,1);						// for AP WiFi 192.168.4.1
	#define defaultChannel 7												// -"-
	#define defaultAuthmode AUTH_OPEN										// -"-

	u8 cur_mode=wifi_get_opmode();
	u8 *myp = params, mode=indexE(&myp, 10);								// get sub-type
	u8 my_ssid[32]={0}, my_pw[64]={0};
	struct ip_info myIP={0};
	bool USE_DHCP=TRUE;
	const char *HELP="{5,1/2=St/AP,...";

	if(find_char(params, '?')) {
		switch (mode){
			case 1:{
				HELP="{5,1,\"ssid\",\"pword[>7]\",(optional)\"IP.IP.IP.IP\",\"GW.GW.GW.GW\",\"NM.NM.NM.NM\"}";
			}
			break;
			case 2:{
				HELP="{5,2,\"ssid\",\"pword[>7]\",(optional)\"IP.IP.IP.IP\",auth_mode=0/4,channel=1/13}";
			}
			break;
		}
		toP0((u8*) HELP,0,0); return;
	}
	if(mode==0 || mode>2) return;											// if sub-type=Station, AP, both
	myp++;
	Ex(&myp,0);																// point to "ssid"
	if(out$.len==0) return;
	fl_memcpy(my_ssid,out$.val,out$.len);
	my_ssid[out$.len]=0;
	myp++;																	// pw may be empty
	Ex(&myp,0);
	if(out$.len==0) my_memset(my_pw, 0, sizeof(my_pw));
	fl_memcpy(my_pw,out$.val,out$.len);
	my_pw[out$.len=0];
	if (*myp>0 && *(myp + 1) == 34){															// if follows IP
		u8 *p=out$.val, IP1,IP2,IP3,IP4;
		myp++;
		Ex(&myp,0);
		out$.val[out$.len]=0;
		IP1= indexE(&p, 10);
		p++;
		IP2= indexE(&p, 10);
		p++;
		IP3= indexE(&p, 10);
		p++;
		IP4= indexE(&p, 10);
		IP4_ADDR(&myIP.ip,IP1,IP2,IP3,IP4);
		USE_DHCP=FALSE;
		if (*myp>0 && *(myp + 1) == 34){														// if follows GW
			p=out$.val;
			myp++;
			Ex(&myp,0);
			out$.val[out$.len]=0;
			IP1= indexE(&p, 10);
			p++;
			IP2= indexE(&p, 10);
			p++;
			IP3= indexE(&p, 10);
			p++;
			IP4= indexE(&p, 10);
			IP4_ADDR(&myIP.gw,IP1,IP2,IP3,IP4);
			if (*myp>0 && *(myp + 1) == 34){													// if follows NM
				p=out$.val;
				myp++;
				Ex(&myp,0);
				out$.val[out$.len]=0;
				IP1= indexE(&p, 10);
				p++;
				IP2= indexE(&p, 10);
				p++;
				IP3= indexE(&p, 10);
				p++;
				IP4= indexE(&p, 10);
				IP4_ADDR(&myIP.netmask,IP1,IP2,IP3,IP4);
			}
			else{
				IP4_ADDR(&myIP.netmask,255,255,255,0);
			}
		}
		else{
			IP4_ADDR(&myIP.gw,0,0,0,0);
			IP4_ADDR(&myIP.netmask,255,255,255,0);
		}
	}
	else if(mode==2){
		defaultIP;
		IP4_ADDR(&myIP.gw,0,0,0,0);
		IP4_ADDR(&myIP.netmask,255,255,255,0);
	}

	//================
	if(mode==1){																// if Station (mode=1)
		struct station_config stationConf;
		if(wifi_station_dhcpc_status()==DHCP_STARTED) wifi_station_dhcpc_stop();
		if(wifi_station_get_connect_status()>STATION_IDLE) wifi_station_disconnect();
			TRACE_DEBUG("intf=%d,mode=%d,DHCP=%d,ssid=%s,pw=%s\r\n",intf_num,cur_mode,USE_DHCP,my_ssid,my_pw);
		if((cur_mode & 1)==0) wifi_set_opmode(cur_mode | 1);                                               // Set station mode
		wifi_station_get_config(&stationConf);
		os_memset(stationConf.ssid, 0, sizeof(stationConf.ssid));
		os_memset(stationConf.password, 0, sizeof(stationConf.password));
		fl_memcpy(stationConf.ssid, my_ssid, my_strlen(my_ssid));
		fl_memcpy(stationConf.password, my_pw, my_strlen(my_pw));
		stationConf.bssid_set=0;
		if(USE_DHCP) wifi_station_dhcpc_start();
		else wifi_set_ip_info(STATION_IF,&myIP);
		wifi_station_set_config(&stationConf);                              	// Set wifi conf
		wifi_station_connect();
		network_init(&myInterfaces[intf_num].state);
		myInterfaces[intf_num].state=INTF_CREATE;
	}
	else{																		// if AP (mode=2)
		struct softap_config apConf={0};
		wifi_set_opmode(cur_mode | 2);                                               // Set AP mode
		if(wifi_softap_dhcps_status()==DHCP_STARTED) wifi_softap_dhcps_stop();
		wifi_softap_get_config(&apConf);
		apConf.ssid_len=0;
		apConf.ssid_hidden=0;
		os_memset(apConf.ssid, 0, sizeof(apConf.ssid));
		os_memset(apConf.password, 0, sizeof(apConf.password));
		fl_memcpy(apConf.ssid, my_ssid,my_strlen(my_ssid));
		fl_memcpy(apConf.password, my_pw,my_strlen(my_pw));

		if(*myp>0) apConf.authmode=indexE(&myp,10);
		else apConf.authmode=defaultAuthmode;

		if(*myp>0) apConf.channel=indexE(&myp,10);
		else apConf.channel=defaultChannel;

		wifi_set_ip_info(SOFTAP_IF,&myIP);
		wifi_softap_set_config(&apConf);
		wifi_softap_dhcps_start();
		myInterfaces[intf_num].state=INTF_OPEN;
	}

	  myInterfaces[intf_num].id=mode-1;									// id=mode-1 i.e. match to index 0/1 of pipe array when mode =1/2
	  myInterfaces[intf_num].closeInterface =wifi_close;				// call ex: x=(*uart0_close)(0)
	  myInterfaces[intf_num].sendInterface=wifi_send;
	  myInterfaces[intf_num].openPipe=wifiAddPipe;
	  myInterfaces[intf_num].closePipe=wifiRemovePipe;
	  myInterfaces[intf_num].type=5;
	  myInterfaces[intf_num].subtype=mode;
	  myInterfaces[intf_num].pipes_attached=0;

}
