/**=====================================
	UART0
=======================================*/
#include "globals.h"
#include "uart.h"

UartDevice UartDev;											// Uart1

struct {
	s8 pipe_id;												/// index of pipe
	u8 intf_num;											/// number of interface pipe attached to
}LOCAL pipesUart0[PIPE_COUNT]={-1,0};				/// Every pipe goes to its index in this array + pipesAttachedUart[port][pipe#]

LOCAL STATUS uart_tx_one_char(uint8 , uint8);
/******************************************************************************
 * FunctionName : uart0_rx_intr_handler
 * Description  : Internal used function
 *                UART0 interrupt handler, add self handle code inside
 * Parameters   : void *para - point to ETS_UART_INTR_ATTACH's arg
 * Returns      : NONE
*******************************************************************************/



LOCAL void uart0_rx_intr_handler(void *param)
{
    /* uart0 and uart1 intr combine together, when interrupt occur, see reg 0x3ff20020, bit2, bit0 represents
     * uart1 and uart0 respectively - not done here
     */
	RcvMsgBuff *pRxBuff = (RcvMsgBuff *)param;
    u8 i, RcvChar;
    s8 doRock=-1;


    if (UART_RXFIFO_FULL_INT_ST != (READ_PERI_REG(UART_INT_ST(UART0)) & UART_RXFIFO_FULL_INT_ST)) {
        return;
    }

    WRITE_PERI_REG(UART_INT_CLR(UART0), UART_RXFIFO_FULL_INT_CLR);

    while (READ_PERI_REG(UART_STATUS(UART0)) & (UART_RXFIFO_CNT << UART_RXFIFO_CNT_S)) {
        RcvChar = READ_PERI_REG(UART_FIFO(UART0)) & 0xFF;

        //TRACE_DEBUG("%c",RcvChar);

        // Fill all the rcvBuffers of pipes attached (no attributes to check) with chars received
    	for (i=0;i<PIPE_COUNT;i++){
    		s8 pipe=pipesUart0[i].pipe_id;
    		if(pipe>=0){
    			if (RcvChar == 8 && mode.f.L==0 && *myStreams[pipe].rxPipe.len > 0) (*myStreams[pipe].rxPipe.len)--;	// if backspace
    			else{
    				myStreams[pipe].rxPipe.val[*myStreams[pipe].rxPipe.len]=RcvChar;
    				(*myStreams[pipe].rxPipe.len)++;
    			}
   	    		if(pipe==0) doRock=i;														// catch Pipe0 index to resolve interface number
    		}
    	}
    }
	if(doRock>=0){																			// if Console
		if(rock(pipesUart0[doRock].intf_num)){
			*myStreams[0].rxPipe.len = 0;
			ReqPipe0.intf=pipesUart0[doRock].intf_num;
			ReqPipe0.i_pipe=doRock;
		}
	}
}

/******************************************************************************
 * FunctionName : uart1_tx_one_char
 * Description  : Internal used function
 *                Use uart1 interface to transfer one char
 * Parameters   : uint8 TxChar - character to tx
 * Returns      : OK
*******************************************************************************/
LOCAL STATUS uart_tx_one_char(uint8 uart, uint8 TxChar)
{
    while (true)
    {
      uint32 fifo_cnt = READ_PERI_REG(UART_STATUS(uart)) & (UART_TXFIFO_CNT<<UART_TXFIFO_CNT_S);
      if ((fifo_cnt >> UART_TXFIFO_CNT_S & UART_TXFIFO_CNT) < 126) {
        break;
      }
    }

    WRITE_PERI_REG(UART_FIFO(uart) , TxChar);
    return OK;
}

/******************************************************************************
 * FunctionName : uart1_write_char
 * Description  : Internal used function
 *                Do some special deal while tx char is '\r' or '\n'
 * Parameters   : char c - character to tx
 * Returns      : NONE
*******************************************************************************/
LOCAL void ICACHE_FLASH_ATTR uart0_write_char(char c)
{
  {
    uart_tx_one_char(UART0, c);
  }
}

/******************************************************************************
 * FunctionName : uart0_tx_buffer
 * Description  : find a pipe attached with non-zero length and transmit it trough uart0
 * Parameters   : char *s - point to send buffer
 *
 * Returns      :
*******************************************************************************/
void ICACHE_FLASH_ATTR uart_send(u8 intf_num)
{
	u8 i, *s;
	if(myInterfaces[intf_num].id==0){
		for (i=0;i<PIPE_COUNT;i++){
			if(pipesUart0[i].pipe_id>0 || (pipesUart0[i].pipe_id==0 && pipesUart0[i].intf_num==ReqPipe0.intf && i==ReqPipe0.i_pipe)){
				if(pipesUart0[i].intf_num==intf_num && *myStreams[pipesUart0[i].pipe_id].txPipe.len>0){
				  uint16 j;
				  for (j = 0; j < *myStreams[pipesUart0[i].pipe_id].txPipe.len; j++)
				  {
					uart_tx_one_char(0, myStreams[pipesUart0[i].pipe_id].txPipe.val[j]);
				  }
				  *myStreams[pipesUart0[i].pipe_id].txPipe.len=0;
				  break;
				}
			}
		}
	}
 }

/******************************************************************************
 * FunctionName : uartAddPipe
 * Description  : attach a pipe to interface
 * Parameters   : u8 pipe, u8 intf_number
 *
 * Returns      :
*******************************************************************************/
void ICACHE_FLASH_ATTR uartAddPipe(u8 pipe, u8 intf_num, u8 *params){
	u8 i;
	const char *HELP="{interfaceN}";

	if(find_char(params, '?')) {toP0((u8*) HELP,0,0); return;}
	if(myInterfaces[intf_num].state>0){
		for(i=0; i<PIPE_COUNT; i++){
			if(pipesUart0[i].pipe_id<0){									// find one empty pipe
				if(!createPipe(pipe)) {toP0("Can't create pipe %i",pipe,0); return;}													// malloc pipe buffers
				pipesUart0[i].pipe_id=pipe;									// set pipe_id to new pipe
				pipesUart0[i].intf_num=intf_num;									// set pipe_id to new pipe
				myInterfaces[intf_num].pipes_attached++;
				myInterfaces[intf_num].state = INTF_CONNECT;
				break;
			}
		}
	}
}
/******************************************************************************
 * FunctionName : uartRemovePipe
 * Description  : remove a pipe from interface
 * Parameters   : u8 pipe, u8 intf_number
 *
 * Returns      :
*******************************************************************************/
void ICACHE_FLASH_ATTR uartRemovePipe(u8 pipe, u8 intf_num){
	u8 i;
	for(i=0; i<PIPE_COUNT; i++){
		if(pipesUart0[i].pipe_id==pipe && pipesUart0[i].intf_num==intf_num){
			pipesUart0[i].pipe_id=-1;
			freePipeCheck(pipe);
			if(--(myInterfaces[intf_num].pipes_attached)<=0){
				myInterfaces[intf_num].pipes_attached=0;
				myInterfaces[intf_num].state=INTF_OPEN;
			}
		}
	}
}
/******************************************************************************
 * FunctionName : uart_close
 * Description  : remove all the pipes attached and close interface
 * Parameters   : u8 intf_number
 *
 * Returns      :
*******************************************************************************/
void ICACHE_FLASH_ATTR uart_close(u8 intf_num){
	u8 i;
	for (i=0;i<PIPE_COUNT;i++){
		uartRemovePipe(i, intf_num);
	}
	myInterfaces[intf_num].state=INTF_CLOSE;
}

/******************************************************************************
 * FunctionName : uart_config
 * Description  : Internal used function
 *                UART0 used for data TX/RX, RX buffer size is 0x100, interrupt enabled
 *                UART1 just used for debug output
 * Parameters   : uart_no, use UART0 or UART1 defined ahead
 * Returns      : NONE
*******************************************************************************/
void  ICACHE_FLASH_ATTR initSerial(u8 intf_num, u8 *params) //parameters=sub_type, uart_number, u16 baud rate/100, bits_parity_stopB , hd)							// Interface_number intfN:, subType, baud rate, bits, parity, stop bits, HD flow contr.
{
	u8 *myp = params, st;
	u8 uart_no, bits, par, sb, hd, i;
	u16 br;
	const char *HELP="{1=Serial, 1=rs232, hd_Num=0, baud_rate/100, 8N1, hd_flow_c=0}";

	if(find_char(params, '?')) {toP0((u8*) HELP,0,0); return;}
	st = indexE(&myp, 10);							// get sub-type
	if (st == 1){									// if sub-type=RS232
		uart_no = indexE(&myp, 10);					// interface number
		br = indexE(&myp, 10)*100;					// baud rate/100
		bits = indexE(&myp, 10);					// data bits
		switch (*myp){
		case 'N':{par = 0; break; }					// parity NONE
		case 'E':{par = 1; break; }					// parity EVEN
		case 'O':{par = 2; break; }					// parity ODD
		}
		myp++;
		sb = indexE(&myp, 10);						// stop bits
		hd = indexE(&myp, 10);						// hardware flow control Y/N=1/0
	}

	//UartDev.baut_rate=9600;
	UartDev.data_bits=bits-5;
	UartDev.stop_bits=sb-1;
	UartDev.exist_parity = (par>0)? STICK_PARITY_EN : STICK_PARITY_DIS;
	UartDev.parity= (par==1)? EVEN_BITS : NONE_BITS;
	UartDev.buff_uart_no=uart_no;
	UartDev.flow_ctrl=(UartFlowCtrl) hd;

  if (uart_no == UART1)
  {
    PIN_FUNC_SELECT(PERIPHS_IO_MUX_GPIO2_U, FUNC_U1TXD_BK);
  }
  else
  {
    /* rcv_buff size if 0x100 */
    ETS_UART_INTR_ATTACH(uart0_rx_intr_handler,  &(UartDev.rcv_buff));	//
    PIN_PULLUP_DIS(PERIPHS_IO_MUX_U0TXD_U);
    PIN_FUNC_SELECT(PERIPHS_IO_MUX_U0TXD_U, FUNC_U0TXD);
//    PIN_FUNC_SELECT(PERIPHS_IO_MUX_MTDO_U, FUNC_U0RTS);
  }

  uart_div_modify(uart_no, UART_CLK_FREQ / (br));

  WRITE_PERI_REG(UART_CONF0(uart_no), UartDev.exist_parity
                 | UartDev.parity
                 | (UartDev.stop_bits << UART_STOP_BIT_NUM_S)
                 | (UartDev.data_bits << UART_BIT_NUM_S));

  //clear rx and tx fifo,not ready
  SET_PERI_REG_MASK(UART_CONF0(uart_no), UART_RXFIFO_RST | UART_TXFIFO_RST);
  CLEAR_PERI_REG_MASK(UART_CONF0(uart_no), UART_RXFIFO_RST | UART_TXFIFO_RST);

  //set rx fifo trigger
//  WRITE_PERI_REG(UART_CONF1(uart_no),
//                 ((UartDev.rcv_buff.TrigLvl & UART_RXFIFO_FULL_THRHD) << UART_RXFIFO_FULL_THRHD_S) |
//                 ((96 & UART_TXFIFO_EMPTY_THRHD) << UART_TXFIFO_EMPTY_THRHD_S) |
//                 UART_RX_FLOW_EN);
  if (uart_no == UART0)
  {
    //set rx fifo trigger
    WRITE_PERI_REG(UART_CONF1(uart_no),
                   ((0x01 & UART_RXFIFO_FULL_THRHD) << UART_RXFIFO_FULL_THRHD_S) |
                   ((0x01 & UART_RX_FLOW_THRHD) << UART_RX_FLOW_THRHD_S) |
                   UART_RX_FLOW_EN);
  }
  else
  {
    WRITE_PERI_REG(UART_CONF1(uart_no),
                   ((UartDev.rcv_buff.TrigLvl & UART_RXFIFO_FULL_THRHD) << UART_RXFIFO_FULL_THRHD_S));
  }

  //clear all interrupt
  WRITE_PERI_REG(UART_INT_CLR(uart_no), 0xffff);
  //enable rx_interrupt
  SET_PERI_REG_MASK(UART_INT_ENA(uart_no), UART_RXFIFO_FULL_INT_ENA);
  // install uart0 putc callback
  os_install_putc1((void *)uart0_write_char);
  //ETS_UART_INTR_ATTACH(uart0_rx_intr_handler,  &(UartDev.rcv_buff));
  //ETS_UART_INTR_ATTACH(uart0_rx_intr_handler, (void) *param);

  myInterfaces[intf_num].id=uart_no;									// RS Port0
  myInterfaces[intf_num].closeInterface =uart_close;					// call ex: x=(*uart0_close)(0)
  myInterfaces[intf_num].sendInterface=uart_send;
  myInterfaces[intf_num].openPipe=uartAddPipe;
  myInterfaces[intf_num].closePipe=uartRemovePipe;
  myInterfaces[intf_num].type=1;
  myInterfaces[intf_num].subtype=1;
  myInterfaces[intf_num].state=INTF_OPEN;
  myInterfaces[intf_num].pipes_attached=0;

  for (i=0;i<PIPE_COUNT;i++){
	  pipesUart0[i].pipe_id=-1;
  }
  ETS_UART_INTR_ENABLE();
 }
//========================================================================================================
